/*
 * StunEffectInfo.h
 *
 * Created 8/30/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/30/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef STUN_EFFECT_INFO_H
 #define STUN_EFFECT_INFO_H
 
 template <typename T> class StunEffectInfo;
 
 #include "global.h"
 
 #include "collisionEntry.h"
 #include "EffectInfo.h"
 #include "EffectTask.h"
 #include "pointerTo.h"
 
 #include "StunInfo.h"
 
 /**
  * Class specification for StunEffectInfo
  */
 template <typename T>
 class StunEffectInfo : virtual public EffectInfo<T>
 {
 // Protected Static Functions
 protected:
    static inline void process_stun_effect( EffectInfo<T>* effect_info_Ptr, Object<T>* into_object_Ptr, const CollisionEntry& c_entry );
 
 // Data Members
 private:
    double _stun_duration;
    
 // Local Functions
 public:
    StunEffectInfo( const double stun_duration, EffectFunc* effect_func = &StunEffectInfo<T>::process_stun_effect );
    StunEffectInfo( const StunEffectInfo<T>& stun_effect_info );
    virtual ~StunEffectInfo();
    inline StunEffectInfo<T>& operator=( const StunEffectInfo<T>& stun_effect_info );
    inline double get_stun_duration() const;
    inline void set_stun_duration( const double stun_duration );
 
 // Private Functions
 private:
    
 // Static Functions
 public:
    
 };
 
 /** STATIC FUNCTIONS **/
 
 /**
  * process_stun_effect() applies the specified stun effect pointed to onto the specified object pointed to.
  * The effect_info_Ptr must be a pointer to a StunEffectInfo.
  *
  * @param (EffectInfo<T>*) effect_info_Ptr
  * @param (Object<T>*) into_object_Ptr
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void StunEffectInfo<T>::process_stun_effect( EffectInfo<T>* effect_info_Ptr, Object<T>* into_object_Ptr, 
                                                     const CollisionEntry& c_entry )
 {  
    nassertv( effect_info_Ptr != NULL );
    nassertv( into_object_Ptr != NULL );
    
    StunEffectInfo<T>* stun_effect_info_Ptr( dynamic_cast<StunEffectInfo<T>*>(effect_info_Ptr) );
    
    PT(StunInfo<T>) stun_info( new StunInfo<T>( stun_effect_info_Ptr->get_stun_duration() ) );
    PT(EffectTask<T>) effect_task( new EffectTask<T>( into_object_Ptr, stun_info ) );
    effect_task->add_to_task_manager();
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 StunEffectInfo<T>::StunEffectInfo( const double stun_duration, EffectFunc* effect_func )
                   : EffectInfo<T>( effect_func ),
                     _stun_duration( stun_duration )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 StunEffectInfo<T>::StunEffectInfo( const StunEffectInfo<T>& stun_effect_info )
                   : EffectInfo<T>( stun_effect_info ),
                     _stun_duration( stun_effect_info._stun_duration )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 StunEffectInfo<T>::~StunEffectInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified StunEffectInfo to this StunEffectInfo.
  *
  * @param (const StunEffectInfo<T>&) stun_effect_info
  * @return StunEffectInfo<T>&
  */
 template <typename T>
 inline StunEffectInfo<T>& StunEffectInfo<T>::operator=( const StunEffectInfo<T>& stun_effect_info )
 {
    EffectInfo<T>::operator=( stun_effect_info );
    _stun_duration = stun_effect_info._stun_duration;
    
    return *this;
 }
 
 /**
  * get_stun_duration() returns the stun duration.
  *
  * @return double
  */
 template <typename T>
 inline double StunEffectInfo<T>::get_stun_duration() const
 {
    return _stun_duration;
 }
 
 /**
  * set_stun_duration() sets the stun duration to the specified stun_duration.
  *
  * @param (const double) stun_duration
  */
 template <typename T>
 inline void StunEffectInfo<T>::set_stun_duration( const double stun_duration )
 {
    _stun_duration = stun_duration;
 }
 
 /** STATIC FUNCTIONS **/
 
 #endif // STUN_EFFECT_INFO_H